import React, { useState, useEffect } from 'react';
import { Link, useLocation } from 'react-router-dom';
import './SidebarNav.css'; // UNCOMMENTED

interface SidebarNavItem {
  name: string;
  path: string;
  children?: SidebarNavItem[];
}

// This data structure will eventually define your site's hierarchy
// We will add open/close state and logic to this component later
const sidebarNavigationLinks: SidebarNavItem[] = [
  { name: "Home", path: "/home" },
  { name: "About", path: "/about" },
  {
    name: "Videos",
    path: "/videos",
    children: [
      {
        name: "Video Map",
        path: "/videomap",
        children: [
          { name: "Video 1a", path: "/videomap/video1a" },
          {
            name: "Video 2a",
            path: "/videomap/video2a",
            children: [{ name: "Video 2b", path: "/videomap/video2b" }],
          },
        ],
      },
      { name: "Guided", path: "/videos/guided"}
    ],
  },
  { name: "Contact", path: "/contact" },
];

// Helper function to find all parent paths of a given activePath
const getParentPaths = (items: SidebarNavItem[], activePath: string): string[] => {
  const paths: string[] = [];
  function findPath(currentItems: SidebarNavItem[], currentSearchPath: string) {
    for (const item of currentItems) {
      if (currentSearchPath.startsWith(item.path)) {
        if (item.children && item.children.length > 0) {
          paths.push(item.path);
          findPath(item.children, currentSearchPath);
        } else if (item.path === currentSearchPath) {
          // Found the exact item or a leaf node matching the start
          return true;
        }
      }
    }
    return false;
  }
  findPath(items, activePath);
  return paths;
};

const renderSidebarNavItems = (
  items: SidebarNavItem[], 
  currentPath: string,
  openItems: Set<string>,
  toggleItem: (path: string) => void
) => {
  return (
    <ul className="theme-sidebar-links">
      {items.map(item => {
        const isOpen = openItems.has(item.path);
        const hasChildren = item.children && item.children.length > 0;
        return (
          <li key={item.path} className={`${currentPath === item.path ? 'active-sidebar-link' : ''} ${hasChildren ? 'has-children' : ''}`}>
            <div className="sidebar-item-container">
              <Link to={item.path}>{item.name}</Link>
              {hasChildren && (
                <span className="folder-toggle" onClick={() => toggleItem(item.path)}>
                  {isOpen ? '[-] ' : '[+] '}
                </span>
              )}
            </div>
            {hasChildren && isOpen && item.children && (
              renderSidebarNavItems(item.children, currentPath, openItems, toggleItem)
            )}
          </li>
        );
      })}
    </ul>
  );
};

const SidebarNav: React.FC = () => {
  const location = useLocation();
  const [openItems, setOpenItems] = useState<Set<string>>(new Set());

  useEffect(() => {
    // Determine which items should be open based on the current path
    const pathsToOpen = getParentPaths(sidebarNavigationLinks, location.pathname);
    // Add the current path itself if it's a parent, to ensure its direct children are shown if it has any
    // This might not be strictly necessary if getParentPaths correctly identifies all parents whose children might contain the active link.
    // For simplicity, we ensure all ancestors of the active path are open.
    const currentItemIsParent = sidebarNavigationLinks.some(item => item.path === location.pathname && item.children);
    if (currentItemIsParent) {
        pathsToOpen.push(location.pathname);
    }
    
    // A more robust way to ensure the direct parent of an active leaf node is open:
    // Find the item that IS the current path. If it has children, it should be open.
    // Or find the parent of the current path.
    let initialOpen = new Set<string>(pathsToOpen);

    // If current path is a leaf, its parent should be open.
    // This logic can be intricate. getParentPaths aims to handle opening ancestors.
    // We might need to refine getParentPaths or add logic here if a direct parent isn't being opened.

    sidebarNavigationLinks.forEach(item => {
        if (location.pathname.startsWith(item.path) && item.path !== location.pathname && item.children){
            const segments = location.pathname.split('/').filter(s => s);
            const itemSegments = item.path.split('/').filter(s => s);
            if (segments.length > itemSegments.length && location.pathname.startsWith(item.path)){
                // This means item is an ancestor of a deeper path
                // This is mostly handled by getParentPaths, kept for potential refinement
            }
        }
        // If the current path is an exact match to a parent item, open it
        if (item.path === location.pathname && item.children && item.children.length > 0) {
            initialOpen.add(item.path);
        }
    });

    setOpenItems(initialOpen);

  }, [location.pathname]);

  const toggleItem = (path: string) => {
    setOpenItems(prevOpenItems => {
      const newOpenItems = new Set(prevOpenItems);
      if (newOpenItems.has(path)) {
        newOpenItems.delete(path);
      } else {
        newOpenItems.add(path);
      }
      return newOpenItems;
    });
  };

  return (
    <nav aria-label="Sidebar Navigation">
      {renderSidebarNavItems(sidebarNavigationLinks, location.pathname, openItems, toggleItem)}
    </nav>
  );
};

export default SidebarNav; 